
import React, { useState, useEffect } from 'react';
import { motion } from 'framer-motion';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { RefreshCw, TrendingUp, DollarSign, ArrowLeft, ShoppingCart } from 'lucide-react';
import TransaksiForm from '@/components/TransaksiForm';
import TransaksiTable from '@/components/TransaksiTable';
import { getTransaksiToday } from '@/lib/supabase';

const TransaksiPage = ({ onBack }) => {
  const [refreshTrigger, setRefreshTrigger] = useState(0);
  const [todaySummary, setTodaySummary] = useState({ count: 0, total: 0 });

  const handleSuccess = () => {
    setRefreshTrigger(prev => prev + 1);
  };

  const handleRefresh = () => {
    setRefreshTrigger(prev => prev + 1);
  };

  useEffect(() => {
    const fetchTodaySummary = async () => {
      try {
        const { data, error } = await getTransaksiToday();
        if (!error && data) {
          const count = data.length;
          const total = data.reduce((sum, t) => sum + parseFloat(t.total_harga || 0), 0);
          setTodaySummary({ count, total });
        }
      } catch (error) {
        console.error('Error fetching today summary:', error);
      }
    };

    fetchTodaySummary();
  }, [refreshTrigger]);

  const formatRupiah = (amount) => {
    return new Intl.NumberFormat('id-ID', {
      style: 'currency',
      currency: 'IDR',
      minimumFractionDigits: 0,
    }).format(amount);
  };

  return (
    <div className="space-y-6">
      <motion.div
        initial={{ opacity: 0, y: -10 }}
        animate={{ opacity: 1, y: 0 }}
        className="flex items-center justify-between"
      >
        <div className="flex items-center gap-4">
          {onBack && (
            <Button variant="ghost" size="icon" onClick={onBack} className="text-blue-100 hover:text-white hover:bg-blue-800">
              <ArrowLeft className="h-6 w-6" />
            </Button>
          )}
          <div>
            <h1 className="text-2xl font-bold text-white flex items-center gap-2">
              <ShoppingCart className="h-6 w-6" />
              Kasir & Transaksi
            </h1>
            <p className="text-blue-200 text-sm">Catat penjualan baru</p>
          </div>
        </div>
      </motion.div>

      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.3 }}
        className="grid grid-cols-1 md:grid-cols-2 gap-4"
      >
        <Card className="bg-gradient-to-br from-blue-500 to-blue-600 border-0 text-white shadow-lg">
          <CardHeader className="pb-3">
            <CardTitle className="text-sm font-medium text-blue-100 flex items-center">
              <TrendingUp className="h-4 w-4 mr-2" />
              Transaksi Hari Ini
            </CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-3xl font-bold">{todaySummary.count}</p>
            <p className="text-sm text-blue-100 mt-1">Total transaksi tercatat</p>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-green-500 to-green-600 border-0 text-white shadow-lg">
          <CardHeader className="pb-3">
            <CardTitle className="text-sm font-medium text-green-100 flex items-center">
              <DollarSign className="h-4 w-4 mr-2" />
              Total Penjualan Hari Ini
            </CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-3xl font-bold">{formatRupiah(todaySummary.total)}</p>
            <p className="text-sm text-green-100 mt-1">Nilai total transaksi</p>
          </CardContent>
        </Card>
      </motion.div>

      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.3, delay: 0.1 }}
      >
        <TransaksiForm onSuccess={handleSuccess} />
      </motion.div>

      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.3, delay: 0.2 }}
        className="flex justify-end"
      >
        <Button
          onClick={handleRefresh}
          variant="outline"
          className="border-blue-300 text-blue-600 hover:bg-blue-50"
        >
          <RefreshCw className="h-4 w-4 mr-2" />
          Refresh Data
        </Button>
      </motion.div>

      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.3, delay: 0.3 }}
      >
        <TransaksiTable refreshTrigger={refreshTrigger} />
      </motion.div>
    </div>
  );
};

export default TransaksiPage;
