
import React, { useState } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import { useAuth } from '@/contexts/AuthContext';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { Label } from '@/components/ui/label';
import { useToast } from '@/components/ui/use-toast';
import { Loader2, UserPlus, AlertCircle } from 'lucide-react';
import { isSupabaseConfigured } from '@/lib/supabase';

const RegisterPage = () => {
  const { register } = useAuth();
  const navigate = useNavigate();
  const { toast } = useToast();
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState({
    username: '',
    email: '',
    nomor_telepon: '',
    password: '',
    confirmPassword: ''
  });

  const handleChange = (e) => {
    setFormData({...formData, [e.target.name]: e.target.value});
  };

  const validate = () => {
    if (formData.password.length < 8) return "Password minimal 8 karakter";
    if (formData.password !== formData.confirmPassword) return "Password tidak cocok";
    if (!formData.email.includes('@')) return "Format email tidak valid";
    if (!formData.nomor_telepon.match(/^(\+62|0)[0-9]{8,}$/)) return "Format nomor telepon tidak valid";
    return null;
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    
    if (!isSupabaseConfigured()) {
      toast({
        title: "Konfigurasi Error",
        description: "Supabase belum dikonfigurasi.",
        variant: "destructive"
      });
      return;
    }

    const errorMsg = validate();
    if (errorMsg) {
      toast({ title: "Validasi Error", description: errorMsg, variant: "destructive" });
      return;
    }

    setLoading(true);
    try {
      const { success, error } = await register({
        username: formData.username,
        email: formData.email,
        nomor_telepon: formData.nomor_telepon,
        password: formData.password
      });

      if (success) {
        toast({
          title: "Registrasi Berhasil",
          description: "Silahkan login dengan akun baru anda",
        });
        navigate('/login');
      } else {
        toast({
          title: "Registrasi Gagal",
          description: error || "Terjadi kesalahan saat mendaftar",
          variant: "destructive"
        });
      }
    } catch (err) {
      console.error("Register exception:", err);
      toast({
        title: "System Error",
        description: "Terjadi kesalahan tak terduga.",
        variant: "destructive"
      });
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-slate-900 via-blue-900 to-slate-800 p-4">
      <Card className="w-full max-w-md bg-white/95 backdrop-blur-sm shadow-xl">
        <CardHeader>
          <CardTitle className="text-2xl font-bold text-gray-900">Daftar Akun Baru</CardTitle>
          <CardDescription>Bergabung dengan Bakulan sekarang</CardDescription>
        </CardHeader>
        <CardContent>
          {!isSupabaseConfigured() && (
            <div className="mb-4 p-3 bg-red-100 border border-red-300 rounded flex items-start gap-2 text-red-700 text-sm">
              <AlertCircle className="h-4 w-4 mt-0.5 flex-shrink-0" />
              <p>Database belum terhubung. Pendaftaran ditutup sementara.</p>
            </div>
          )}
          <form onSubmit={handleSubmit} className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="username">Username</Label>
              <Input id="username" name="username" value={formData.username} onChange={handleChange} required className="bg-white" disabled={loading} />
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="email">Email</Label>
                <Input id="email" name="email" type="email" value={formData.email} onChange={handleChange} required className="bg-white" disabled={loading} />
              </div>
              <div className="space-y-2">
                <Label htmlFor="nomor_telepon">No. Telepon</Label>
                <Input id="nomor_telepon" name="nomor_telepon" value={formData.nomor_telepon} onChange={handleChange} required className="bg-white" placeholder="08..." disabled={loading} />
              </div>
            </div>
            <div className="space-y-2">
              <Label htmlFor="password">Password</Label>
              <Input id="password" name="password" type="password" value={formData.password} onChange={handleChange} required className="bg-white" placeholder="Min 8 karakter" disabled={loading} />
            </div>
            <div className="space-y-2">
              <Label htmlFor="confirmPassword">Konfirmasi Password</Label>
              <Input id="confirmPassword" name="confirmPassword" type="password" value={formData.confirmPassword} onChange={handleChange} required className="bg-white" disabled={loading} />
            </div>
            <Button type="submit" className="w-full bg-blue-600 hover:bg-blue-700 text-white" disabled={loading || !isSupabaseConfigured()}>
              {loading ? (
                <><Loader2 className="mr-2 h-4 w-4 animate-spin" /> Mendaftar...</>
              ) : (
                <><UserPlus className="mr-2 h-4 w-4" /> Daftar Sekarang</>
              )}
            </Button>
          </form>
        </CardContent>
        <CardFooter className="flex justify-center text-sm">
          <p className="text-gray-600">
            Sudah punya akun? <Link to="/login" className="text-blue-600 hover:underline font-semibold">Login disini</Link>
          </p>
        </CardFooter>
      </Card>
    </div>
  );
};

export default RegisterPage;
