
import React, { useState } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import { useAuth } from '@/contexts/AuthContext';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { Label } from '@/components/ui/label';
import { useToast } from '@/components/ui/use-toast';
import { Loader2, LogIn, Store, AlertCircle } from 'lucide-react';
import { isSupabaseConfigured } from '@/lib/supabase';

const LoginPage = () => {
  const { login } = useAuth();
  const navigate = useNavigate();
  const { toast } = useToast();
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState({
    identifier: '',
    password: ''
  });

  const handleSubmit = async (e) => {
    e.preventDefault();
    
    if (!isSupabaseConfigured()) {
      toast({
        title: "Konfigurasi Error",
        description: "Supabase belum dikonfigurasi. Cek .env file.",
        variant: "destructive"
      });
      return;
    }

    if (!formData.identifier || !formData.password) {
      toast({
        title: "Data Tidak Lengkap",
        description: "Mohon isi email/telepon dan password.",
        variant: "destructive"
      });
      return;
    }

    setLoading(true);

    try {
      const { success, error } = await login(formData.identifier, formData.password);

      if (success) {
        toast({
          title: "Login Berhasil",
          description: "Selamat datang kembali di Bakulan!",
        });
        navigate('/menu');
      } else {
        toast({
          title: "Login Gagal",
          description: error || "Periksa kembali data anda",
          variant: "destructive"
        });
      }
    } catch (err) {
      console.error("Login exception:", err);
      toast({
        title: "Error",
        description: "Terjadi kesalahan sistem.",
        variant: "destructive"
      });
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-slate-900 via-blue-900 to-slate-800 p-4">
      <Card className="w-full max-w-md bg-white/95 backdrop-blur-sm shadow-xl">
        <CardHeader className="text-center">
          <div className="flex justify-center mb-4">
            <div className="h-16 w-16 bg-blue-100 rounded-full flex items-center justify-center">
              <Store className="h-10 w-10 text-blue-600" />
            </div>
          </div>
          <CardTitle className="text-2xl font-bold text-gray-900">Bakulan Login</CardTitle>
          <CardDescription>Masuk untuk mengelola dagangan anda</CardDescription>
        </CardHeader>
        <CardContent>
          {!isSupabaseConfigured() && (
            <div className="mb-4 p-3 bg-red-100 border border-red-300 rounded flex items-start gap-2 text-red-700 text-sm">
              <AlertCircle className="h-4 w-4 mt-0.5 flex-shrink-0" />
              <p>Database belum terhubung. Aplikasi tidak dapat memproses login.</p>
            </div>
          )}
          <form onSubmit={handleSubmit} className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="identifier">Email atau No. Telepon</Label>
              <Input 
                id="identifier" 
                placeholder="user@example.com / 0812..." 
                value={formData.identifier}
                onChange={(e) => setFormData({...formData, identifier: e.target.value})}
                required
                className="bg-white"
                disabled={loading}
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="password">Password</Label>
              <Input 
                id="password" 
                type="password"
                placeholder="••••••••"
                value={formData.password}
                onChange={(e) => setFormData({...formData, password: e.target.value})}
                required
                className="bg-white"
                disabled={loading}
              />
            </div>
            <Button type="submit" className="w-full bg-blue-600 hover:bg-blue-700 text-white" disabled={loading || !isSupabaseConfigured()}>
              {loading ? (
                <><Loader2 className="mr-2 h-4 w-4 animate-spin" /> Masuk...</>
              ) : (
                <><LogIn className="mr-2 h-4 w-4" /> Masuk</>
              )}
            </Button>
          </form>
        </CardContent>
        <CardFooter className="flex justify-center text-sm">
          <p className="text-gray-600">
            Belum punya akun? <Link to="/register" className="text-blue-600 hover:underline font-semibold">Daftar disini</Link>
          </p>
        </CardFooter>
      </Card>
    </div>
  );
};

export default LoginPage;
