
import React, { createContext, useContext, useState, useEffect } from 'react';
import { loginUser as loginService, registerUser as registerService, isSupabaseConfigured } from '@/lib/supabase';

const AuthContext = createContext(null);

export const AuthProvider = ({ children }) => {
  const [user, setUser] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const initAuth = async () => {
      // 1. Check if configured
      if (!isSupabaseConfigured()) {
        console.warn("Auth disabled: Supabase not configured");
        setLoading(false);
        return;
      }

      // 2. Restore session
      const storedUser = localStorage.getItem('bakulan_user');
      if (storedUser) {
        try {
          const parsedUser = JSON.parse(storedUser);
          if (parsedUser && parsedUser.id) {
            setUser(parsedUser);
          } else {
            localStorage.removeItem('bakulan_user');
          }
        } catch (e) {
          console.error('Failed to parse user session', e);
          localStorage.removeItem('bakulan_user');
        }
      }
      setLoading(false);
    };

    initAuth();
  }, []);

  const login = async (identifier, password) => {
    if (!isSupabaseConfigured()) {
      return { success: false, error: "Database belum dikonfigurasi. Hubungi admin." };
    }

    setLoading(true);
    try {
      const { data, error } = await loginService(identifier, password);
      
      if (error) {
        throw new Error(error.message || "Login gagal");
      }
      
      if (!data) {
        throw new Error("Data user tidak ditemukan");
      }

      setUser(data);
      localStorage.setItem('bakulan_user', JSON.stringify(data));
      return { success: true };
    } catch (error) {
      console.error("Login Error:", error);
      return { success: false, error: error.message };
    } finally {
      setLoading(false);
    }
  };

  const register = async (userData) => {
    if (!isSupabaseConfigured()) {
      return { success: false, error: "Database belum dikonfigurasi. Hubungi admin." };
    }

    setLoading(true);
    try {
      const { data, error } = await registerService(userData);
      
      if (error) {
        throw new Error(error.message || "Registrasi gagal");
      }
      
      return { success: true, data };
    } catch (error) {
      console.error("Register Error:", error);
      return { success: false, error: error.message };
    } finally {
      setLoading(false);
    }
  };

  const logout = () => {
    setUser(null);
    localStorage.removeItem('bakulan_user');
  };

  return (
    <AuthContext.Provider value={{ user, login, register, logout, loading, isAuthenticated: !!user }}>
      {children}
    </AuthContext.Provider>
  );
};

export const useAuth = () => {
  const context = useContext(AuthContext);
  if (!context) {
    throw new Error('useAuth must be used within an AuthProvider');
  }
  return context;
};
