
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { useToast } from '@/components/ui/use-toast';
import { getTransaksi, isSupabaseConfigured } from '@/lib/supabase';
import { Loader2, Receipt, ArrowUpDown } from 'lucide-react';
import { motion } from 'framer-motion';
import { Button } from '@/components/ui/button';

const TransaksiTable = ({ refreshTrigger }) => {
  const { toast } = useToast();
  const [loading, setLoading] = useState(true);
  const [transaksi, setTransaksi] = useState([]);
  const [sortOrder, setSortOrder] = useState('desc'); // 'asc' or 'desc'

  const fetchTransaksi = async () => {
    setLoading(true);
    try {
      if (!isSupabaseConfigured()) {
        throw new Error("Supabase belum dikonfigurasi");
      }

      const { data, error } = await getTransaksi();
      
      if (error) throw error;
      
      setTransaksi(data || []);
    } catch (error) {
      console.error('Error fetching transaksi:', error);
      if (error.message !== "Supabase belum dikonfigurasi") {
        toast({
          title: "Error",
          description: "Gagal mengambil data transaksi",
          variant: "destructive",
        });
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchTransaksi();
  }, [refreshTrigger]);

  const toggleSortOrder = () => {
    setSortOrder(prev => prev === 'desc' ? 'asc' : 'desc');
  };

  const sortedTransaksi = [...transaksi].sort((a, b) => {
    const dateA = new Date(a.tanggal_transaksi);
    const dateB = new Date(b.tanggal_transaksi);
    return sortOrder === 'desc' ? dateB - dateA : dateA - dateB;
  });

  const formatRupiah = (amount) => {
    return new Intl.NumberFormat('id-ID', {
      style: 'currency',
      currency: 'IDR',
      minimumFractionDigits: 0,
    }).format(amount);
  };

  const formatDateTime = (dateString) => {
    return new Date(dateString).toLocaleString('id-ID', {
      day: 'numeric',
      month: 'short',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  if (loading) {
    return (
      <Card className="bg-white/95 backdrop-blur-sm border-blue-200">
        <CardContent className="flex items-center justify-center py-12">
          <Loader2 className="h-8 w-8 animate-spin text-blue-600" />
        </CardContent>
      </Card>
    );
  }

  if (transaksi.length === 0) {
    return (
      <Card className="bg-white/95 backdrop-blur-sm border-blue-200">
        <CardContent className="flex flex-col items-center justify-center py-12">
          <Receipt className="h-12 w-12 text-gray-400 mb-4" />
          <p className="text-gray-600 text-center">Belum ada transaksi tercatat</p>
          <p className="text-gray-400 text-sm text-center mt-2">
            Catat transaksi pertama Anda untuk memulai
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card className="bg-white/95 backdrop-blur-sm border-blue-200">
      <CardHeader>
        <div className="flex items-center justify-between">
          <div>
            <CardTitle className="text-gray-900">Riwayat Transaksi</CardTitle>
            <CardDescription>
              Total {transaksi.length} transaksi tercatat
            </CardDescription>
          </div>
          <Button
            variant="outline"
            size="sm"
            onClick={toggleSortOrder}
            className="border-gray-300 text-gray-900"
          >
            <ArrowUpDown className="h-4 w-4 mr-2" />
            {sortOrder === 'desc' ? 'Terbaru' : 'Terlama'}
          </Button>
        </div>
      </CardHeader>
      <CardContent>
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead>
              <tr className="border-b border-gray-200">
                <th className="text-left py-3 px-4 text-sm font-semibold text-gray-900">Nama Barang</th>
                <th className="text-center py-3 px-4 text-sm font-semibold text-gray-900">Jumlah</th>
                <th className="text-right py-3 px-4 text-sm font-semibold text-gray-900">Harga Satuan</th>
                <th className="text-right py-3 px-4 text-sm font-semibold text-gray-900">Total Harga</th>
                <th className="text-left py-3 px-4 text-sm font-semibold text-gray-900">Tanggal</th>
              </tr>
            </thead>
            <tbody>
              {sortedTransaksi.map((item, index) => (
                <motion.tr
                  key={item.id}
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: index * 0.03 }}
                  className="border-b border-gray-100 hover:bg-blue-50/50 transition-colors"
                >
                  <td className="py-3 px-4 text-gray-900 font-medium">
                    {item.inventori?.nama_barang || 'N/A'}
                  </td>
                  <td className="py-3 px-4 text-center text-gray-900">
                    {item.jumlah}
                  </td>
                  <td className="py-3 px-4 text-right text-gray-900">
                    {formatRupiah(item.harga_satuan)}
                  </td>
                  <td className="py-3 px-4 text-right text-gray-900 font-semibold">
                    {formatRupiah(item.total_harga)}
                  </td>
                  <td className="py-3 px-4 text-gray-600 text-sm">
                    {formatDateTime(item.tanggal_transaksi)}
                  </td>
                </motion.tr>
              ))}
            </tbody>
          </table>
        </div>
      </CardContent>
    </Card>
  );
};

export default TransaksiTable;
