import React, { useState } from 'react';
import { motion } from 'framer-motion';
import { Search, Trash2, Calendar, DollarSign } from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { useToast } from '@/components/ui/use-toast';

const TransactionHistory = ({ transactions, onDeleteTransaction }) => {
  const { toast } = useToast();
  const [searchTerm, setSearchTerm] = useState('');

  const filteredTransactions = transactions.filter(transaction =>
    transaction.customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
    transaction.items.some(item => item.name.toLowerCase().includes(searchTerm.toLowerCase()))
  );

  const handleDelete = (id) => {
    onDeleteTransaction(id);
    toast({
      title: "Transaksi Dihapus",
      description: "Transaksi berhasil dihapus dari riwayat",
    });
  };

  const formatDate = (timestamp) => {
    const date = new Date(timestamp);
    return date.toLocaleDateString('id-ID', {
      day: 'numeric',
      month: 'long',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  return (
    <motion.div
      initial={{ opacity: 0, scale: 0.95 }}
      animate={{ opacity: 1, scale: 1 }}
      transition={{ duration: 0.3 }}
    >
      <Card className="bg-slate-800/50 backdrop-blur-sm border-slate-700">
        <CardHeader>
          <CardTitle className="text-2xl text-white flex items-center gap-2">
            <Calendar className="w-6 h-6 text-blue-400" />
            Riwayat Transaksi
          </CardTitle>
          <div className="relative mt-4">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-slate-400 w-5 h-5" />
            <Input
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              placeholder="Cari transaksi..."
              className="pl-10 bg-slate-700/50 border-slate-600 text-white placeholder:text-slate-400"
            />
          </div>
        </CardHeader>
        <CardContent>
          {filteredTransactions.length === 0 ? (
            <div className="text-center py-12">
              <p className="text-slate-400 text-lg">
                {searchTerm ? 'Tidak ada transaksi yang ditemukan' : 'Belum ada transaksi'}
              </p>
            </div>
          ) : (
            <div className="space-y-4">
              {filteredTransactions.map((transaction, index) => (
                <motion.div
                  key={transaction.id}
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ duration: 0.3, delay: index * 0.05 }}
                  className="bg-slate-700/30 rounded-lg p-4 border border-slate-600 hover:border-blue-500/50 transition-all"
                >
                  <div className="flex justify-between items-start mb-3">
                    <div>
                      <h3 className="text-white font-semibold text-lg">
                        {transaction.customerName}
                      </h3>
                      <p className="text-slate-400 text-sm flex items-center gap-1">
                        <Calendar className="w-3 h-3" />
                        {formatDate(transaction.timestamp)}
                      </p>
                    </div>
                    <Button
                      variant="destructive"
                      size="icon"
                      onClick={() => handleDelete(transaction.id)}
                      className="hover:scale-110 transition-transform"
                    >
                      <Trash2 className="w-4 h-4" />
                    </Button>
                  </div>

                  <div className="space-y-2 mb-3">
                    {transaction.items.map((item, idx) => (
                      <div key={idx} className="flex justify-between text-sm">
                        <span className="text-slate-300">
                          {item.name} x{item.quantity}
                        </span>
                        <span className="text-slate-300">
                          Rp {(item.quantity * item.price).toLocaleString('id-ID')}
                        </span>
                      </div>
                    ))}
                  </div>

                  <div className="flex justify-between items-center pt-3 border-t border-slate-600">
                    <span className="text-slate-400 text-sm capitalize">
                      {transaction.paymentMethod === 'cash' ? 'Tunai' : transaction.paymentMethod === 'debit' ? 'Debit' : 'QRIS'}
                    </span>
                    <div className="flex items-center gap-2">
                      <DollarSign className="w-5 h-5 text-blue-400" />
                      <span className="text-blue-300 font-bold text-lg">
                        Rp {transaction.total.toLocaleString('id-ID')}
                      </span>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>
    </motion.div>
  );
};

export default TransactionHistory;