import React, { useState } from 'react';
import { motion } from 'framer-motion';
import { Plus, Trash2, ShoppingBag } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { useToast } from '@/components/ui/use-toast';

const TransactionForm = ({ onAddTransaction }) => {
  const { toast } = useToast();
  const [items, setItems] = useState([{ name: '', quantity: 1, price: 0 }]);
  const [customerName, setCustomerName] = useState('');
  const [paymentMethod, setPaymentMethod] = useState('cash');

  const addItem = () => {
    setItems([...items, { name: '', quantity: 1, price: 0 }]);
  };

  const removeItem = (index) => {
    if (items.length > 1) {
      setItems(items.filter((_, i) => i !== index));
    }
  };

  const updateItem = (index, field, value) => {
    const newItems = [...items];
    newItems[index][field] = value;
    setItems(newItems);
  };

  const calculateTotal = () => {
    return items.reduce((sum, item) => sum + (item.quantity * item.price), 0);
  };

  const handleSubmit = (e) => {
    e.preventDefault();
    
    if (items.some(item => !item.name || item.price <= 0)) {
      toast({
        title: "Error",
        description: "Mohon lengkapi semua item dengan benar",
        variant: "destructive",
      });
      return;
    }

    const transaction = {
      customerName: customerName || 'Pelanggan Umum',
      items: items.filter(item => item.name),
      total: calculateTotal(),
      paymentMethod,
    };

    onAddTransaction(transaction);
    
    setItems([{ name: '', quantity: 1, price: 0 }]);
    setCustomerName('');
    setPaymentMethod('cash');

    toast({
      title: "Transaksi Berhasil! 🎉",
      description: `Total: Rp ${calculateTotal().toLocaleString('id-ID')}`,
    });
  };

  return (
    <motion.div
      initial={{ opacity: 0, scale: 0.95 }}
      animate={{ opacity: 1, scale: 1 }}
      transition={{ duration: 0.3 }}
    >
      <Card className="bg-slate-800/50 backdrop-blur-sm border-slate-700">
        <CardHeader>
          <CardTitle className="text-2xl text-white flex items-center gap-2">
            <ShoppingBag className="w-6 h-6 text-blue-400" />
            Transaksi Baru
          </CardTitle>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleSubmit} className="space-y-6">
            <div className="space-y-2">
              <Label htmlFor="customerName" className="text-white">Nama Pelanggan</Label>
              <Input
                id="customerName"
                value={customerName}
                onChange={(e) => setCustomerName(e.target.value)}
                placeholder="Masukkan nama pelanggan (opsional)"
                className="bg-slate-700/50 border-slate-600 text-white placeholder:text-slate-400"
              />
            </div>

            <div className="space-y-4">
              <div className="flex justify-between items-center">
                <Label className="text-white text-lg">Item Belanja</Label>
                <Button
                  type="button"
                  onClick={addItem}
                  size="sm"
                  className="bg-blue-600 hover:bg-blue-700"
                >
                  <Plus className="w-4 h-4 mr-1" />
                  Tambah Item
                </Button>
              </div>

              {items.map((item, index) => (
                <motion.div
                  key={index}
                  initial={{ opacity: 0, x: -20 }}
                  animate={{ opacity: 1, x: 0 }}
                  transition={{ duration: 0.3 }}
                  className="grid grid-cols-12 gap-3 items-end bg-slate-700/30 p-4 rounded-lg"
                >
                  <div className="col-span-5">
                    <Label className="text-white text-sm">Nama Item</Label>
                    <Input
                      value={item.name}
                      onChange={(e) => updateItem(index, 'name', e.target.value)}
                      placeholder="Nama produk"
                      className="bg-slate-700/50 border-slate-600 text-white placeholder:text-slate-400"
                      required
                    />
                  </div>
                  <div className="col-span-3">
                    <Label className="text-white text-sm">Jumlah</Label>
                    <Input
                      type="number"
                      min="1"
                      value={item.quantity}
                      onChange={(e) => updateItem(index, 'quantity', parseInt(e.target.value) || 1)}
                      className="bg-slate-700/50 border-slate-600 text-white"
                      required
                    />
                  </div>
                  <div className="col-span-3">
                    <Label className="text-white text-sm">Harga</Label>
                    <Input
                      type="number"
                      min="0"
                      value={item.price}
                      onChange={(e) => updateItem(index, 'price', parseFloat(e.target.value) || 0)}
                      placeholder="0"
                      className="bg-slate-700/50 border-slate-600 text-white"
                      required
                    />
                  </div>
                  <div className="col-span-1">
                    <Button
                      type="button"
                      variant="destructive"
                      size="icon"
                      onClick={() => removeItem(index)}
                      disabled={items.length === 1}
                      className="w-full"
                    >
                      <Trash2 className="w-4 h-4" />
                    </Button>
                  </div>
                </motion.div>
              ))}
            </div>

            <div className="space-y-2">
              <Label className="text-white">Metode Pembayaran</Label>
              <div className="grid grid-cols-3 gap-3">
                {['cash', 'debit', 'qris'].map((method) => (
                  <button
                    key={method}
                    type="button"
                    onClick={() => setPaymentMethod(method)}
                    className={`p-3 rounded-lg border-2 transition-all ${
                      paymentMethod === method
                        ? 'border-blue-500 bg-blue-600/20 text-blue-300'
                        : 'border-slate-600 bg-slate-700/30 text-slate-300 hover:border-slate-500'
                    }`}
                  >
                    {method === 'cash' ? 'Tunai' : method === 'debit' ? 'Debit' : 'QRIS'}
                  </button>
                ))}
              </div>
            </div>

            <div className="bg-blue-900/30 p-4 rounded-lg border border-blue-700/50">
              <div className="flex justify-between items-center">
                <span className="text-xl text-white font-semibold">Total:</span>
                <span className="text-3xl text-blue-300 font-bold">
                  Rp {calculateTotal().toLocaleString('id-ID')}
                </span>
              </div>
            </div>

            <Button
              type="submit"
              className="w-full bg-gradient-to-r from-blue-600 to-blue-700 hover:from-blue-700 hover:to-blue-800 text-white text-lg py-6"
            >
              Proses Transaksi
            </Button>
          </form>
        </CardContent>
      </Card>
    </motion.div>
  );
};

export default TransactionForm;