
import React, { useState } from 'react';
import { Card, CardContent } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from '@/components/ui/dialog';
import { deletePelanggan, isSupabaseConfigured } from '@/lib/supabase';
import { useToast } from '@/components/ui/use-toast';
import { Pencil, Trash2, Users, Loader2 } from 'lucide-react';
import { motion } from 'framer-motion';

const PelangganTable = ({ data, loading, onEdit, onRefresh }) => {
  const { toast } = useToast();
  const [deleteDialog, setDeleteDialog] = useState({ open: false, pelanggan: null });
  const [deleting, setDeleting] = useState(false);

  const handleDelete = async () => {
    if (!deleteDialog.pelanggan) return;

    if (!isSupabaseConfigured()) {
      toast({
        title: "Error",
        description: "Supabase belum dikonfigurasi",
        variant: "destructive",
      });
      return;
    }

    setDeleting(true);
    try {
      const { error } = await deletePelanggan(deleteDialog.pelanggan.id);
      
      if (error) throw error;

      toast({
        title: "Berhasil!",
        description: "Pelanggan berhasil dihapus",
      });

      setDeleteDialog({ open: false, pelanggan: null });
      if (onRefresh) onRefresh();
    } catch (error) {
      console.error('Error deleting pelanggan:', error);
      toast({
        title: "Error",
        description: "Gagal menghapus pelanggan",
        variant: "destructive",
      });
    } finally {
      setDeleting(false);
    }
  };

  if (loading) {
    return (
      <Card className="bg-white/95 backdrop-blur-sm border-blue-200">
        <CardContent className="flex items-center justify-center py-12">
          <Loader2 className="h-8 w-8 animate-spin text-blue-600" />
        </CardContent>
      </Card>
    );
  }

  if (!data || data.length === 0) {
    return (
      <Card className="bg-white/95 backdrop-blur-sm border-blue-200">
        <CardContent className="flex flex-col items-center justify-center py-12">
          <Users className="h-12 w-12 text-gray-400 mb-4" />
          <p className="text-gray-600 text-center">Belum ada data pelanggan</p>
          <p className="text-gray-400 text-sm text-center mt-2">
            Tambahkan pelanggan baru untuk mulai mengelola
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <>
      <Card className="bg-white/95 backdrop-blur-sm border-blue-200">
        <CardContent className="p-0">
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead>
                <tr className="border-b border-gray-200 bg-blue-50/50">
                  <th className="text-left py-3 px-4 text-sm font-semibold text-gray-900">Nama</th>
                  <th className="text-left py-3 px-4 text-sm font-semibold text-gray-900">No. Telepon</th>
                  <th className="text-left py-3 px-4 text-sm font-semibold text-gray-900">Alamat</th>
                  <th className="text-center py-3 px-4 text-sm font-semibold text-gray-900">Aksi</th>
                </tr>
              </thead>
              <tbody>
                {data.map((item, index) => (
                  <motion.tr
                    key={item.id}
                    initial={{ opacity: 0, y: 10 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: index * 0.05 }}
                    className="border-b border-gray-100 hover:bg-blue-50/30 transition-colors"
                  >
                    <td className="py-3 px-4">
                      <div className="font-medium text-gray-900">{item.nama_pelanggan}</div>
                      <div className="text-xs text-gray-500">{item.email}</div>
                    </td>
                    <td className="py-3 px-4 text-gray-700">{item.nomor_telepon}</td>
                    <td className="py-3 px-4 text-gray-700 max-w-xs truncate">{item.alamat}</td>
                    <td className="py-3 px-4">
                      <div className="flex items-center justify-center gap-2">
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={() => onEdit(item)}
                          className="h-8 w-8 p-0 text-blue-600 hover:text-blue-700 hover:bg-blue-50 rounded-full"
                        >
                          <Pencil className="h-4 w-4" />
                        </Button>
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={() => setDeleteDialog({ open: true, pelanggan: item })}
                          className="h-8 w-8 p-0 text-red-600 hover:text-red-700 hover:bg-red-50 rounded-full"
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </td>
                  </motion.tr>
                ))}
              </tbody>
            </table>
          </div>
        </CardContent>
      </Card>

      <Dialog open={deleteDialog.open} onOpenChange={(open) => setDeleteDialog({ open, pelanggan: null })}>
        <DialogContent className="bg-white">
          <DialogHeader>
            <DialogTitle className="text-gray-900">Hapus Pelanggan?</DialogTitle>
            <DialogDescription>
              Apakah Anda yakin ingin menghapus <span className="font-semibold text-gray-900">{deleteDialog.pelanggan?.nama_pelanggan}</span>?
              Tindakan ini tidak dapat dibatalkan.
            </DialogDescription>
          </DialogHeader>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setDeleteDialog({ open: false, pelanggan: null })}
              disabled={deleting}
              className="border-gray-300 text-gray-900"
            >
              Batal
            </Button>
            <Button
              onClick={handleDelete}
              disabled={deleting}
              className="bg-red-600 hover:bg-red-700 text-white"
            >
              {deleting ? (
                <>
                  <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  Menghapus...
                </>
              ) : (
                'Hapus'
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </>
  );
};

export default PelangganTable;
