
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Button } from '@/components/ui/button';
import { useToast } from '@/components/ui/use-toast';
import { createPelanggan, updatePelanggan, isSupabaseConfigured } from '@/lib/supabase';
import { Loader2, UserPlus, Save, X } from 'lucide-react';

const PelangganForm = ({ editingPelanggan, onSuccess, onCancel }) => {
  const { toast } = useToast();
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState({
    nama_pelanggan: '',
    nomor_telepon: '',
    alamat: '',
    email: ''
  });

  const isEditMode = Boolean(editingPelanggan);

  useEffect(() => {
    if (editingPelanggan) {
      setFormData({
        nama_pelanggan: editingPelanggan.nama_pelanggan || '',
        nomor_telepon: editingPelanggan.nomor_telepon || '',
        alamat: editingPelanggan.alamat || '',
        email: editingPelanggan.email || ''
      });
    } else {
      setFormData({ nama_pelanggan: '', nomor_telepon: '', alamat: '', email: '' });
    }
  }, [editingPelanggan]);

  const handleChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    
    if (!isSupabaseConfigured()) {
      toast({
        title: "Error",
        description: "Supabase belum dikonfigurasi",
        variant: "destructive",
      });
      return;
    }

    if (!formData.nama_pelanggan.trim() || !formData.nomor_telepon.trim() || !formData.alamat.trim()) {
      toast({
        title: "Error",
        description: "Nama, No. Telepon, dan Alamat harus diisi",
        variant: "destructive",
      });
      return;
    }

    setLoading(true);

    try {
      let result;
      
      if (isEditMode) {
        result = await updatePelanggan(editingPelanggan.id, formData);
      } else {
        result = await createPelanggan(formData);
      }

      if (result.error) throw result.error;

      toast({
        title: "Berhasil!",
        description: `Pelanggan ${isEditMode ? 'diperbarui' : 'ditambahkan'}`,
      });

      setFormData({ nama_pelanggan: '', nomor_telepon: '', alamat: '', email: '' });
      
      if (onSuccess) onSuccess();
    } catch (error) {
      console.error('Error saving pelanggan:', error);
      toast({
        title: "Error",
        description: error.message || "Gagal menyimpan data",
        variant: "destructive",
      });
    } finally {
      setLoading(false);
    }
  };

  return (
    <Card className="bg-white/95 backdrop-blur-sm border-blue-200">
      <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
        <div className="space-y-1">
          <CardTitle className="text-gray-900">
            {isEditMode ? 'Edit Pelanggan' : 'Tambah Pelanggan Baru'}
          </CardTitle>
          <CardDescription>
            {isEditMode ? 'Perbarui data pelanggan' : 'Daftarkan pelanggan baru'}
          </CardDescription>
        </div>
        {isEditMode && onCancel && (
          <Button variant="ghost" size="icon" onClick={onCancel}>
            <X className="h-4 w-4" />
          </Button>
        )}
      </CardHeader>
      <CardContent>
        <form onSubmit={handleSubmit} className="space-y-4 pt-4">
          <div className="space-y-2">
            <Label htmlFor="nama_pelanggan" className="text-gray-900">Nama Lengkap</Label>
            <Input
              id="nama_pelanggan"
              name="nama_pelanggan"
              placeholder="Contoh: Budi Santoso"
              value={formData.nama_pelanggan}
              onChange={handleChange}
              disabled={loading}
              className="bg-white text-gray-900"
              required
            />
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="nomor_telepon" className="text-gray-900">Nomor Telepon</Label>
              <Input
                id="nomor_telepon"
                name="nomor_telepon"
                placeholder="08123456789"
                value={formData.nomor_telepon}
                onChange={handleChange}
                disabled={loading}
                className="bg-white text-gray-900"
                required
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="email" className="text-gray-900">Email (Opsional)</Label>
              <Input
                id="email"
                name="email"
                type="email"
                placeholder="budi@example.com"
                value={formData.email}
                onChange={handleChange}
                disabled={loading}
                className="bg-white text-gray-900"
              />
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="alamat" className="text-gray-900">Alamat</Label>
            <Input
              id="alamat"
              name="alamat"
              placeholder="Jl. Merdeka No. 45, Jakarta"
              value={formData.alamat}
              onChange={handleChange}
              disabled={loading}
              className="bg-white text-gray-900"
              required
            />
          </div>

          <div className="flex gap-2 pt-2">
            <Button
              type="submit"
              disabled={loading}
              className="flex-1 bg-blue-600 hover:bg-blue-700 text-white"
            >
              {loading ? (
                <>
                  <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  Menyimpan...
                </>
              ) : (
                <>
                  {isEditMode ? (
                    <><Save className="mr-2 h-4 w-4" />Perbarui</>
                  ) : (
                    <><UserPlus className="mr-2 h-4 w-4" />Simpan</>
                  )}
                </>
              )}
            </Button>
            
            {isEditMode && onCancel && (
              <Button
                type="button"
                variant="outline"
                onClick={onCancel}
                disabled={loading}
                className="border-gray-300 text-gray-900"
              >
                Batal
              </Button>
            )}
          </div>
        </form>
      </CardContent>
    </Card>
  );
};

export default PelangganForm;
