
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Button } from '@/components/ui/button';
import { useToast } from '@/components/ui/use-toast';
import { createInventori, updateInventori, isSupabaseConfigured } from '@/lib/supabase';
import { Loader2, Plus, Save } from 'lucide-react';

const InventoriForm = ({ selectedBarang, onSuccess, onCancel }) => {
  const { toast } = useToast();
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState({
    nama_barang: '',
    harga: '',
    stok: ''
  });

  const isEditMode = Boolean(selectedBarang);

  useEffect(() => {
    if (selectedBarang) {
      setFormData({
        nama_barang: selectedBarang.nama_barang || '',
        harga: selectedBarang.harga || '',
        stok: selectedBarang.stok || ''
      });
    } else {
      setFormData({ nama_barang: '', harga: '', stok: '' });
    }
  }, [selectedBarang]);

  const handleChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    
    if (!isSupabaseConfigured()) {
      toast({
        title: "Error",
        description: "Supabase belum dikonfigurasi",
        variant: "destructive",
      });
      return;
    }

    if (!formData.nama_barang.trim() || !formData.harga || !formData.stok) {
      toast({
        title: "Error",
        description: "Semua field harus diisi",
        variant: "destructive",
      });
      return;
    }

    if (parseFloat(formData.harga) < 0 || parseInt(formData.stok) < 0) {
      toast({
        title: "Error",
        description: "Harga dan stok tidak boleh negatif",
        variant: "destructive",
      });
      return;
    }

    setLoading(true);

    try {
      let result;
      
      if (isEditMode) {
        result = await updateInventori(selectedBarang.id, formData);
      } else {
        result = await createInventori(formData);
      }

      if (result.error) {
        throw result.error;
      }

      toast({
        title: "Berhasil!",
        description: `Barang ${isEditMode ? 'diperbarui' : 'ditambahkan'} ke inventori`,
      });

      setFormData({ nama_barang: '', harga: '', stok: '' });
      
      if (onSuccess) {
        onSuccess();
      }
    } catch (error) {
      console.error('Error saving inventori:', error);
      toast({
        title: "Error",
        description: error.message || "Gagal menyimpan data ke database",
        variant: "destructive",
      });
    } finally {
      setLoading(false);
    }
  };

  return (
    <Card className="bg-white/95 backdrop-blur-sm border-blue-200">
      <CardHeader>
        <CardTitle className="text-gray-900">
          {isEditMode ? 'Edit Barang' : 'Tambah Barang Baru'}
        </CardTitle>
        <CardDescription>
          {isEditMode ? 'Perbarui data barang di inventori' : 'Tambahkan barang baru ke inventori'}
        </CardDescription>
      </CardHeader>
      <CardContent>
        <form onSubmit={handleSubmit} className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="nama_barang" className="text-gray-900">Nama Barang</Label>
            <Input
              id="nama_barang"
              name="nama_barang"
              type="text"
              placeholder="Contoh: Mie Instan"
              value={formData.nama_barang}
              onChange={handleChange}
              disabled={loading}
              className="bg-white text-gray-900 placeholder:text-gray-400"
              required
            />
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="harga" className="text-gray-900">Harga (Rp)</Label>
              <Input
                id="harga"
                name="harga"
                type="number"
                placeholder="0"
                value={formData.harga}
                onChange={handleChange}
                disabled={loading}
                className="bg-white text-gray-900 placeholder:text-gray-400"
                min="0"
                step="0.01"
                required
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="stok" className="text-gray-900">Stok</Label>
              <Input
                id="stok"
                name="stok"
                type="number"
                placeholder="0"
                value={formData.stok}
                onChange={handleChange}
                disabled={loading}
                className="bg-white text-gray-900 placeholder:text-gray-400"
                min="0"
                required
              />
            </div>
          </div>

          <div className="flex gap-2">
            <Button
              type="submit"
              disabled={loading}
              className="flex-1 bg-blue-600 hover:bg-blue-700 text-white"
            >
              {loading ? (
                <>
                  <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  Menyimpan...
                </>
              ) : (
                <>
                  {isEditMode ? (
                    <><Save className="mr-2 h-4 w-4" />Perbarui</>
                  ) : (
                    <><Plus className="mr-2 h-4 w-4" />Tambah</>
                  )}
                </>
              )}
            </Button>
            
            {isEditMode && onCancel && (
              <Button
                type="button"
                variant="outline"
                onClick={onCancel}
                disabled={loading}
                className="border-gray-300 text-gray-900"
              >
                Batal
              </Button>
            )}
          </div>
        </form>
      </CardContent>
    </Card>
  );
};

export default InventoriForm;
